/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.utils;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.format.DateTimeFormatter;
import java.time.format.ResolverStyle;
import java.time.temporal.ChronoField;
import java.time.temporal.IsoFields;
import java.time.temporal.TemporalAccessor;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import eu.europa.ec.taxud.cesop.domain.DocTypeEnum;
import eu.europa.ec.taxud.cesop.domain.MessageTypeEnum;
import eu.europa.ec.taxud.cesop.domain.MessageTypeIndicEnum;
import eu.europa.ec.taxud.cesop.domain.TransactionDateEnum;
import eu.europa.ec.taxud.cesop.domain.ValidationError;
import eu.europa.ec.taxud.cesop.domain.ValidationErrorType;
import eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum;
import eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeHolder;
import eu.europa.ec.taxud.cesop.domain.XmlCountryTypeAndValue;
import eu.europa.ec.taxud.cesop.domain.XmlPsp;
import eu.europa.ec.taxud.cesop.domain.XmlTransactionDate;
import eu.europa.ec.taxud.cesop.domain.XmlTypeAndValue;

import static eu.europa.ec.taxud.cesop.domain.DocTypeEnum.CESOP1;
import static eu.europa.ec.taxud.cesop.domain.DocTypeEnum.CESOP2;
import static eu.europa.ec.taxud.cesop.domain.DocTypeEnum.CESOP3;
import static eu.europa.ec.taxud.cesop.domain.MessageTypeEnum.PMT;
import static eu.europa.ec.taxud.cesop.domain.MessageTypeIndicEnum.CESOP100;
import static eu.europa.ec.taxud.cesop.domain.MessageTypeIndicEnum.CESOP101;
import static eu.europa.ec.taxud.cesop.domain.MessageTypeIndicEnum.CESOP102;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_BR_0010;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_BR_0050;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_BR_0060;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_BR_0100;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_BR_0110;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_BR_0120;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_BR_0130;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_BR_0140;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_TR_0020;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_TR_0030;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_TR_0040;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_TR_0050;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_TR_0060;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_TR_0070;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_BR_0150;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_TR_9999;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.MH_BR_0030;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.MH_BR_0070;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.MH_BR_0080;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.MH_BR_0090;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.MH_BR_0100;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.MH_BR_0110;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.MH_BR_0120;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0010;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0020;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0030;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0040;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0050;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0060;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0070;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0080;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0090;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0100;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RP_BR_0110;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RT_BR_0010;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RT_BR_0030;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RT_BR_0040;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RT_BR_0060;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RT_BR_0080;
import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.RT_BR_0090;
import static eu.europa.ec.taxud.cesop.utils.LangUtils.isBlank;
import static eu.europa.ec.taxud.cesop.utils.LangUtils.isNotBlank;
import static java.math.BigDecimal.ZERO;

/**
 * Util class handling error records.
 */
public final class ValidationErrorUtils {

    public static final DateTimeFormatter XML_DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("uuuu-MM-dd'T'HH:mm:ss[.SSS]XXX").withResolverStyle(ResolverStyle.STRICT);

    private static final String XML_TIMESTAMP_FOR_QUARTER_FORMAT = "yyyy-MM";
    private static final DateTimeFormatter XML_DATE_TIME_FOR_QUARTER_FORMAT = DateTimeFormatter.ofPattern(XML_TIMESTAMP_FOR_QUARTER_FORMAT);
    private static final Pattern IBAN_PATTERN = Pattern.compile("[A-Z]{2}[0-9]{2}[0-9A-Za-z]{10,30}");
    private static final BigInteger IBAN_DIVISOR = BigInteger.valueOf(97L);
    private static final BigInteger IBAN_REMAINDER = BigInteger.valueOf(1L);
    private static final Pattern BIC_PATTERN = Pattern.compile("^[a-z]{4}([a-z]{2})[a-z0-9]{2}([a-z0-9]{3})?$", Pattern.CASE_INSENSITIVE);
    private static final String EU_COUNTRY_CODE = "EU";
    private static final String GREECE_EL = "EL";
    private static final String GREECE_GR = "GR";
    private static final String OTHER = "OTHER";
    private static final Set<String> GREECE_COUNTRY_CODES = Stream.of(GREECE_GR, GREECE_EL).collect(Collectors.toSet());
    private static final Set<String> RP_BR_0090_IGNORE_VERSIONS = Stream.of("4.00", "4.01", "4.02").collect(Collectors.toSet());
    private static final Set<String> RP_BR_0100_IGNORE_VERSIONS = Stream.of("4.00", "4.01", "4.02").collect(Collectors.toSet());
    private static final Set<String> RP_BR_0110_IGNORE_VERSIONS = Stream.of("4.00", "4.01", "4.02").collect(Collectors.toSet());
    private static final Set<String> CM_BR_0120_IGNORE_VERSIONS = Stream.of("4.00", "4.01", "4.02").collect(Collectors.toSet());
    private static final Set<String> CM_BR_0130_IGNORE_VERSIONS = Stream.of("4.00", "4.01", "4.02").collect(Collectors.toSet());
    private static final Set<String> CM_BR_0140_IGNORE_VERSIONS = Stream.of("4.00", "4.01", "4.02").collect(Collectors.toSet());

    private ValidationErrorUtils() {
    }

    /**
     * Checks if an initial Payment data message only contains new data and if a correction message only
     * contains corrections/deletions (10070 and 10080).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param messageTypeIndic the message type indic
     * @param docType          the doc type
     * @param docRefId         the doc ref id
     */
    public static void checkMhBr0070AndMhBr0080(final List<ValidationError> validationErrors,
            final String messageRefId, final MessageTypeIndicEnum messageTypeIndic, final DocTypeEnum docType,
            final String docRefId) {
        if (messageTypeIndic == CESOP100 && docType != CESOP1) {
            validationErrors.add(createValidationError(MH_BR_0070, messageRefId, docRefId, null));
        } else if (messageTypeIndic == CESOP101 && docType == CESOP1) {
            validationErrors.add(createValidationError(MH_BR_0080, messageRefId, docRefId, null));
        }
    }

    /**
     * Checks if the DocRefId is not unique in the message (20010).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param docRefIds        the docref ids already checked
     * @param docRefId         the docref id
     */
    public static void checkCmBr0010(final List<ValidationError> validationErrors, final String messageRefId,
            final Set<String> docRefIds, final String docRefId) {
        if (!docRefIds.add(docRefId)) {
            validationErrors.add(createValidationError(CM_BR_0010, messageRefId, docRefId, null));
        }
    }

    /**
     * Checks if the CorrDocRefId is not specified for new data (20050).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param docTypeEnum      the doc type enum
     * @param corrDocRefId     the corr doc ref id
     * @param docRefId         the doc ref id
     */
    public static void checkCmBr0050(final List<ValidationError> validationErrors, final String messageRefId,
            final DocTypeEnum docTypeEnum, final String corrDocRefId, final String docRefId) {
        if (docTypeEnum == CESOP1 && isNotBlank(corrDocRefId)) {
            validationErrors.add(createValidationError(CM_BR_0050, messageRefId, docRefId, null));
        }
    }

    /**
     * Checks if the corrDocRefId is provided in case of correction (20060).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param docTypeEnum      the doc type enum
     * @param corrDocRefId     the corrDocRefId
     * @param docRefId         the doc ref id
     */
    public static void checkCmBr0060(final List<ValidationError> validationErrors, final String messageRefId,
            final DocTypeEnum docTypeEnum, final String corrDocRefId, final String docRefId) {
        if ((docTypeEnum == CESOP2 || docTypeEnum == CESOP3) && isBlank(corrDocRefId)) {
            validationErrors.add(createValidationError(CM_BR_0060, messageRefId, docRefId, null));
        }
    }

    /**
     * Checks if the corrDocRefId references to single message in case of correction (20120).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param docTypeEnum      the doc type enum
     * @param corrDocRefId     the corrDocRefId
     * @param corrDocRefIds    list of corrDocRefIds fetched from related message by CorrMessageRefId
     * @param xsdVersion       the xsd version
     * @param docRefId         the doc ref id
     */
    public static void checkCmBr0120(List<ValidationError> validationErrors, String messageRefId,
            DocTypeEnum docTypeEnum, String corrDocRefId, Set<String> corrDocRefIds, String xsdVersion, String docRefId) {
        if (!CM_BR_0120_IGNORE_VERSIONS.contains(xsdVersion) && (docTypeEnum == CESOP2 || docTypeEnum == CESOP3)
                && corrDocRefId != null && corrDocRefIds != null && !corrDocRefIds.contains(corrDocRefId)) {
            validationErrors.add(createValidationError(CM_BR_0120, messageRefId, docRefId, null));
        }
    }

    /**
     * Checks if the country code of the 'Country' element within the 'ReportedPayee' element is not the same as
     * the country code of the 'PayerMS' element (40010).
     *
     * @param validationErrors      the list of validation errors
     * @param messageRefId          the reference of the message
     * @param payeeCountry          the payee country
     * @param payerMsCountry        the payer MS country
     * @param transactionIdentifier the transaction identifier id
     */
    public static void checkRpBr0010(final List<ValidationError> validationErrors, final String messageRefId,
            final String payeeCountry, final String payerMsCountry, final String transactionIdentifier) {
        if (Objects.equals(payeeCountry, payerMsCountry)) {
            validationErrors.add(createValidationError(RP_BR_0010, messageRefId, null, transactionIdentifier));
        }
    }

    /**
     * Checks the IBAN format (40020).
     *
     * @param validationErrors      the list of validation errors
     * @param messageRefId          the reference of the message
     * @param accountIdentifierType the account identifier type enum
     * @param iban                  the iban
     * @param docRefId              the doc ref id
     * @return the boolean          true if the IBAN is valid, false otherwise
     */
    public static boolean checkRpBr0020(final List<ValidationError> validationErrors, final String messageRefId,
            final String accountIdentifierType, final String iban, final String docRefId) {
        if ("IBAN".equalsIgnoreCase(accountIdentifierType) && (iban == null || !IBAN_PATTERN.matcher(iban).matches())) {
            validationErrors.add(createValidationError(RP_BR_0020, messageRefId, docRefId, null));
            return false;
        }
        return true;
}

    /**
     * Validates the IBAN (40030) based on the countryCode IBAN length and the MOD97 algorithm.
     *
     * @param validationErrors      the list of validation errors
     * @param messageRefId          the reference of the message
     * @param accountIdentifierType the account identifier type enum
     * @param iban                  the iban
     * @param docRefId              the doc ref id
     */
    public static void checkRpBr0030(final List<ValidationError> validationErrors, final String messageRefId,
            final String accountIdentifierType, final String iban,
            final String docRefId) {
        if (!"IBAN".equalsIgnoreCase(accountIdentifierType) || iban == null || iban.length() < 2 || isBlank(iban.substring(0, 2))) {
            return;
        }
        final String countryCode = iban.substring(0, 2);
        final Integer ibanLength = IbanLength.findForCountry(countryCode);
        if (ibanLength == null || ibanLength == 0) {
            // unknown country, country iban set to length zero or uninitialized data holder - can't validate
            return;
        }
        if (iban.length() != ibanLength) {
            validationErrors.add(createValidationError(RP_BR_0030, messageRefId, docRefId, null));
        } else {
            final char[] chars = iban.toCharArray();
            final StringBuilder sb = new StringBuilder(ibanLength * 2);
            for (int index = 4; index < ibanLength; index++) {
                sb.append(Character.getNumericValue(chars[index]));
            }
            for (int index = 0; index < 4; index++) {
                sb.append(Character.getNumericValue(chars[index]));
            }
            final BigInteger ibanInteger = new BigInteger(sb.toString());
            if (ibanInteger.mod(IBAN_DIVISOR).compareTo(IBAN_REMAINDER) != 0) {
                validationErrors.add(createValidationError(RP_BR_0030, messageRefId, docRefId, null));
            }
        }
    }

    /**
     * Checks if no Reported payee is listed in no payment data for the requested period handler (40040).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param hasPaymentData   if the transaction has payment data
     * @param messageTypeIndic the message type indic
     */
    public static void checkRpBr0040(final List<ValidationError> validationErrors, final String messageRefId,
            final boolean hasPaymentData, final MessageTypeIndicEnum messageTypeIndic) {
        if (messageTypeIndic == CESOP102 && hasPaymentData) {
            validationErrors.add(createValidationError(RP_BR_0040, messageRefId));
        }
    }

    /**
     * Checks if the 'ReportedTransaction' element is present, except in case of deletion of the related Reported Payee. (40050).
     *
     * @param validationErrors  the list of validation errors
     * @param messageRefId      the reference of the message
     * @param docType           the doc type of the reported payee
     * @param emptyTransactions true if no reported transactions for this reported payee
     */
    public static void checkRpBr0050(final List<ValidationError> validationErrors, final String messageRefId,
            final DocTypeEnum docType, final boolean emptyTransactions) {
        if (docType != CESOP3 && emptyTransactions) {
            validationErrors.add(createValidationError(RP_BR_0050, messageRefId));
        }
    }

    /**
     * Checks if there is no discrepancy in the 'AccountIdentifier' attributes.
     * (40060)
     *
     * @param validationErrors the validation errors
     * @param account          the account
     * @param messageRefId     the message ref id
     */
    public static void checkRpBr0060(final List<ValidationError> validationErrors, final XmlCountryTypeAndValue account,
            final String messageRefId) {
        if (isNotBlank(account.getValue())) {
            if (isBlank(account.getCountry()) || isBlank(account.getType())) {
                validationErrors.add(createValidationError(RP_BR_0060, messageRefId));
            }
        } else {
            if (isNotBlank(account.getCountry()) || isNotBlank(account.getType())) {
                validationErrors.add(createValidationError(RP_BR_0060, messageRefId));
            }
        }
    }

    /**
     * Checks if the period is not before 01/01/2024 (10030).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param period           the period
     * @param minPeriod        the minimum period allowed
     */
    public static void checkMhBr0030(final List<ValidationError> validationErrors, final String messageRefId,
            final Integer period, final Integer minPeriod) {
        if (period < minPeriod) {
            validationErrors.add(createValidationError(MH_BR_0030, messageRefId));
        }
    }

    /**
     * A payment data message (MessageType = 'PMT') must contain the 'PaymentDataBody' element.
     *
     * @param validationErrors   the list of validation errors
     * @param messageRefId       the reference of the message
     * @param messageType        the message type
     * @param hasPaymentDataBody true if the message contains a 'PaymentDataBody' element, false otherwise
     */
    public static void checkMhBr0090(final List<ValidationError> validationErrors, final String messageRefId,
            final MessageTypeEnum messageType, final boolean hasPaymentDataBody) {
        if (messageType != PMT || !hasPaymentDataBody) {
            validationErrors.add(createValidationError(MH_BR_0090, messageRefId));
        }
    }

    /**
     * In a correction message, the reporting period must be identical to the reporting period of the correlated message.
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the message ref id
     * @param messageTypeIndic the message type indic
     * @param period           the message period
     * @param correlatedPeriod the period of the correlated message
     */
    public static void checkMhBr0100(final List<ValidationError> validationErrors, final String messageRefId,
            final MessageTypeIndicEnum messageTypeIndic, final Integer period, final Integer correlatedPeriod) {
        if (messageTypeIndic == CESOP101 && !period.equals(correlatedPeriod)) {
            validationErrors.add(createValidationError(MH_BR_0100, messageRefId));
        }
    }

    /**
     * CorrMessageRefId in the 'MessageSpec' element must only be provided in correction messages
     * (when 'MessageTypeIndic' = CESOP101). Otherwise, the element must not be provided.
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the message ref id
     * @param corrMessageRefId the CorrMessageRefId value
     * @param messageTypeIndic the message type indic
     */
    public static void checkMhBr0110(final List<ValidationError> validationErrors, final String messageRefId,
            final String corrMessageRefId, final MessageTypeIndicEnum messageTypeIndic) {
        if ((isNotBlank(corrMessageRefId) && messageTypeIndic != CESOP101)
                || (isBlank(corrMessageRefId) && messageTypeIndic == CESOP101)) {
            validationErrors.add(createValidationError(MH_BR_0110, messageRefId));
        }
    }

    /**
     * TransmittingCountry in the 'MessageSpec' element must be the same as required country if that is passed.
     * EU is a wildcard and should allow the upload for any MS.
     *
     * @param validationErrors    the validation errors
     * @param messageRefId        the message ref id
     * @param transmittingCountry the transmitting country
     * @param requiredCountry     the required country
     */
    public static void checkMhBr0120(final List<ValidationError> validationErrors, final String messageRefId,
            final String transmittingCountry, final String requiredCountry) {
        if (isBlank(requiredCountry)) {
            return;
        }
        if (EU_COUNTRY_CODE.equalsIgnoreCase(requiredCountry)) {
            return;
        }
        if (requiredCountry.equalsIgnoreCase(transmittingCountry)) {
            return;
        }
        if (GREECE_COUNTRY_CODES.contains(requiredCountry) && GREECE_COUNTRY_CODES.contains(transmittingCountry)) {
            return;
        }
        validationErrors.add(createValidationError(MH_BR_0120, messageRefId));
    }

    /**
     * Checks the amount value.
     * The refund attribute refers to a wrong value declared in the AmountCurrency element.
     * When the refund attribute is set to 'false', the value of the amount in AmountCurrency element must be positive.
     * When the refund attribute is set to 'true', the value of the amount in the AmountCurrency element must be negative.
     *
     * @param validationErrors      the validation errors
     * @param messageRefId          the message ref id
     * @param isRefund              the is refund
     * @param amount                the amount
     * @param transactionIdentifier the transaction identifier
     */
    public static void checkRtBr0010(final List<ValidationError> validationErrors, final String messageRefId,
            final boolean isRefund, final XmlTypeAndValue amount, final String transactionIdentifier) {
        final BigDecimal decimal = new BigDecimal(amount.getValue());
        if (isRefund && ZERO.compareTo(decimal) < 0 || !isRefund && ZERO.compareTo(decimal) > 0) {
            validationErrors.add(createValidationError(RT_BR_0010, messageRefId, null, transactionIdentifier));
        }
    }

    /**
     * Checks if at least one 'DateTime' element in the 'ReportedTransaction' element refers to a date within the period and year
     * declared in the 'ReportingPeriod' element (45030).
     *
     * @param validationErrors      the list of validation errors
     * @param messageRefId          the reference of the message
     * @param period                the period
     * @param dates                 the transaction dates
     * @param transactionIdentifier the transaction identifier id
     */
    public static void checkRtBr0030(final List<ValidationError> validationErrors, final String messageRefId,
            final Integer period, final EnumMap<TransactionDateEnum, XmlTransactionDate> dates,
            final String transactionIdentifier) {
        for (final XmlTransactionDate date : dates.values()) {
            final String yearMonth = date.getDate().substring(0, XML_TIMESTAMP_FOR_QUARTER_FORMAT.length());
            final TemporalAccessor temporalAccessor = XML_DATE_TIME_FOR_QUARTER_FORMAT.parse(yearMonth);
            final int year = temporalAccessor.get(ChronoField.YEAR);
            final int quarter = temporalAccessor.get(IsoFields.QUARTER_OF_YEAR);
            final int datePeriod = year * 10 + quarter;
            if (period == datePeriod) {
                return;
            }
        }
        validationErrors.add(createValidationError(RT_BR_0030, messageRefId, null, transactionIdentifier));
    }

    /**
     * Checks if the transaction identifier is unique in message handler (45040).
     *
     * @param validationErrors       the list of validation errors
     * @param messageRefId           the reference of the message
     * @param transactionIdentifiers the transaction identifiers ids already checked
     * @param transactionIdentifier  the transaction identifier id
     */
    public static void checkRtBr0040(final List<ValidationError> validationErrors, final String messageRefId,
            final Set<String> transactionIdentifiers, final String transactionIdentifier) {
        if (!transactionIdentifiers.add(transactionIdentifier)) {
            validationErrors.add(createValidationError(RT_BR_0040, messageRefId, null, transactionIdentifier));
        }
    }

    /**
     * Checks if the value of the 'Amount' element is not equal to zero (45060).
     *
     * @param validationErrors      the list of validation errors
     * @param messageRefId          the reference of the message
     * @param amount                the amount
     * @param transactionIdentifier the transaction identifier id
     */
    public static void checkRtBr0060(final List<ValidationError> validationErrors, final String messageRefId,
            final String amount, final String transactionIdentifier) {
        final BigDecimal decimal = new BigDecimal(amount);
        if (ZERO.compareTo(decimal) == 0) {
            validationErrors.add(createValidationError(RT_BR_0060, messageRefId, null, transactionIdentifier));
        }
    }

    /**
     * Checks if the same type of transaction date has not been provided more than once (45080).
     * This rule is checked during the reported transactions parsing.
     *
     * @param validationErrors      the list of validation errors
     * @param messageRefId          the reference of the message
     * @param isInErrorRtBr0080     true if the rule is not respected
     * @param transactionIdentifier the transaction identifier id
     */
    public static void checkRtBr0080(final List<ValidationError> validationErrors, final String messageRefId,
            final boolean isInErrorRtBr0080, final String transactionIdentifier) {
        if (isInErrorRtBr0080) {
            validationErrors.add(createValidationError(RT_BR_0080, messageRefId, null, transactionIdentifier));
        }
    }

    /**
     * Check if the transaction do not provide the corrTransactionIdentifier in case it is not a refund (45090).
     *
     * @param validationErrors          the validation errors
     * @param messageRefId              the message ref id
     * @param isRefund                  the is refund
     * @param transactionIdentifier     the transaction identifier
     * @param corrTransactionIdentifier the corr transaction identifier
     */
    public static void checkRtBr0090(final List<ValidationError> validationErrors, final String messageRefId,
            final boolean isRefund, final String transactionIdentifier, final String corrTransactionIdentifier) {
        if(!isRefund && isNotBlank(corrTransactionIdentifier)){
            validationErrors.add(createValidationError(RT_BR_0090, messageRefId, null, transactionIdentifier));
        }
    }

    /**
     * Failed Decryption (50020).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     */
    public static void checkCmTr0020(final List<ValidationError> validationErrors, final String messageRefId) {
        validationErrors.add(createValidationError(CM_TR_0020, messageRefId));
    }

    /**
     * Failed Decompression (50030).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     */
    public static void checkCmTr0030(final List<ValidationError> validationErrors, final String messageRefId) {
        validationErrors.add(createValidationError(CM_TR_0030, messageRefId));
    }

    /**
     * Failed Signature Check (50040).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     */
    public static void checkCmTr0040(final List<ValidationError> validationErrors, final String messageRefId) {
        validationErrors.add(createValidationError(CM_TR_0040, messageRefId));
    }

    /**
     * Failed Threat Scan (50050).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     */
    public static void checkCmTr0050(final List<ValidationError> validationErrors, final String messageRefId) {
        validationErrors.add(createValidationError(CM_TR_0050, messageRefId));
    }

    /**
     * Failed Virus Scan (50060).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     */
    public static void checkCmTr0060(final List<ValidationError> validationErrors, final String messageRefId) {
        validationErrors.add(createValidationError(CM_TR_0060, messageRefId));
    }

    /**
     * Message size exceeded (50070).
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param size             the size of message
     * @param maxSize          the max size authorized
     */
    public static void checkCmTr0070(final List<ValidationError> validationErrors, final String messageRefId,
            final long size, final long maxSize) {
        if (size > maxSize) {
            validationErrors.add(createValidationError(CM_TR_0070, messageRefId));
        }
    }

    /**
     * Given there's no payees in the message, check if it is allowed for the specified type of message.
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param messageTypeIndic the type of message indic being validated
     */
    public static void checkCmBr0110(final List<ValidationError> validationErrors, final String messageRefId, final MessageTypeIndicEnum messageTypeIndic) {
        if (messageTypeIndic != CESOP101 && messageTypeIndic != CESOP102) {
            validationErrors.add(createValidationError(CM_BR_0110, messageRefId));
        }
    }

    /**
     * Checks if the representative's BIC conforms ISO-9362.
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param representativeId the representative id
     * @param docRefId         the doc ref id
     */
    public static void checkRpBr0070(final List<ValidationError> validationErrors, final String messageRefId, final XmlTypeAndValue representativeId, final String docRefId) {
        checkBicFormat(validationErrors, messageRefId, representativeId, RP_BR_0070, docRefId);
    }

    /**
     * Checks Psp BIC conforms ISO-9362.
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param pspId            the psp id
     */
    public static void checkCmBr0100(final List<ValidationError> validationErrors, final String messageRefId, final XmlTypeAndValue pspId) {
        checkBicFormat(validationErrors, messageRefId, pspId, CM_BR_0100, null);
    }

    /**
     * Checks discrepancy in the provision of an ‘other’ type and its specification, leading to a full rejection
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param type             the type attribute value
     * @param other            the other attribute value
     * @param xsdVersion       the xsd version
     */
    public static void checkCmBr0130(List<ValidationError> validationErrors, String messageRefId, String type, String other, String xsdVersion) {
        if (CM_BR_0130_IGNORE_VERSIONS.contains(xsdVersion)) {
            return;
        }
        if ((OTHER.equalsIgnoreCase(type) && isBlank(other)) || (!OTHER.equalsIgnoreCase(type) && isNotBlank(other))) {
            validationErrors.add(createValidationError(CM_BR_0130, messageRefId));
        }
    }

    /**
     * Checks discrepancy in the provision of an ‘other’ type and its specification, leading to a partial rejection
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param type             the type attribute value
     * @param other            the other attribute value
     * @param xsdVersion       the xsd version
     */
    public static void checkCmBr0140(List<ValidationError> validationErrors, String messageRefId, String type, String other, String xsdVersion) {
        if (CM_BR_0140_IGNORE_VERSIONS.contains(xsdVersion)) {
            return;
        }
        if ((OTHER.equalsIgnoreCase(type) && isBlank(other)) || (!OTHER.equalsIgnoreCase(type) && isNotBlank(other))) {
            validationErrors.add(createValidationError(CM_BR_0140, messageRefId));
        }
    }

    /**
     * For transaction dates.
     * Checks discrepancy in the provision of an ‘other’ type and its specification, leading to a partial rejection.
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the reference of the message
     * @param dates            map of transaction dates
     * @param xsdVersion       the xsd version
     */
    public static void checkCmBr0140ForDates(List<ValidationError> validationErrors, String messageRefId, Map<TransactionDateEnum, XmlTransactionDate> dates, String xsdVersion) {
        if (CM_BR_0140_IGNORE_VERSIONS.contains(xsdVersion)) {
            return;
        }
        if (dates != null) {
            for (Map.Entry<TransactionDateEnum, XmlTransactionDate> entry : dates.entrySet()) {
                if ((TransactionDateEnum.OTHER_DATE.equals(entry.getKey()) && isBlank(entry.getValue().getOther())) ||
                        (!TransactionDateEnum.OTHER_DATE.equals(entry.getKey()) && isNotBlank(entry.getValue().getOther()))) {
                    validationErrors.add(createValidationError(CM_BR_0140, messageRefId));
                    return;
                }
            }
        }
    }

    private static void checkBicFormat(final List<ValidationError> validationErrors, final String messageRefId, final XmlTypeAndValue id, final ValidationErrorTypeEnum errorType, final String docRefId) {
        if ("BIC".equalsIgnoreCase(id.getType())) {
            final Matcher matcher = BIC_PATTERN.matcher(id.getValue());
            if (matcher.matches()) {
                if (IbanLength.findForCountry(matcher.group(1)) == null) {
                    // use IBAN length instead of countries data holder because the first one is statically populated from iban.properties and works in VM
                    validationErrors.add(createValidationError(errorType, messageRefId, docRefId, null));
                }
            } else {
                validationErrors.add(createValidationError(errorType, messageRefId, docRefId, null));
            }
        }
    }

    /**
     * Check rp br 0080.
     *
     * @param validationErrors  the validation errors
     * @param messageRefId      the message ref id
     * @param accountIdentifier the account identifier
     * @param representative    the representative
     * @param docRefId          the doc ref id
     */
    public static void checkRpBr0080(final List<ValidationError> validationErrors, final String messageRefId,
            final XmlCountryTypeAndValue accountIdentifier, final XmlPsp representative, final String docRefId) {
        if ((accountIdentifier == null) == (representative == null)) {
            validationErrors.add(createValidationError(RP_BR_0080, messageRefId, docRefId, null));
        }
    }

    /**
     * Check rp br 0090.
     *
     * @param validationErrors  the validation errors
     * @param messageRefId      the message ref id
     * @param docType           the doc type
     * @param emptyTransactions the empty transactions
     * @param docRefId          the doc ref id
     * @param xsdVersion        the xsd version
     */
    public static void checkRpBr0090(final List<ValidationError> validationErrors, final String messageRefId,
            final DocTypeEnum docType, final boolean emptyTransactions, final String docRefId, final String xsdVersion) {
        if (!RP_BR_0090_IGNORE_VERSIONS.contains(xsdVersion) && DocTypeEnum.CESOP3.equals(docType) && !emptyTransactions) {
            validationErrors.add(createValidationError(RP_BR_0090, messageRefId, docRefId, null));
        }
    }

    /**
     * Check rp br 0100.
     *
     * @param validationErrors the validation errors
     * @param accounts         the accounts
     * @param messageRefId     the message ref id
     * @param docRefId         the doc ref id
     * @param xsdVersion       the xsd version
     */
    public static void checkRpBr0100(final List<ValidationError> validationErrors, final List<XmlCountryTypeAndValue> accounts,
            final String messageRefId, final String docRefId, final String xsdVersion) {
        if (!RP_BR_0100_IGNORE_VERSIONS.contains(xsdVersion)) {
            final int nbBIC = (int) accounts.stream().filter(a -> ("BIC").equalsIgnoreCase(a.getType())).count();
            if(!accounts.isEmpty() && (accounts.size() > 2 || nbBIC > 1 || accounts.size() - nbBIC != 1)){
                validationErrors.add(createValidationError(RP_BR_0100, messageRefId, docRefId, null));
            }
        }
    }

    /**
     * Check that the accountIdentifierOther element, defining the type of account that is reported for
     * "Other" AccountIdentifier types, cannot be filled with IBAN, BIC or OBAN.
     *
     * @param validationErrors the validation errors
     * @param messageRefId     the message ref id
     * @param type             the type
     * @param other            the other
     * @param xsdVersion       the xsd version
     */
    public static void checkRpBr0110(final List<ValidationError> validationErrors, final String messageRefId,
            final String type, final String other, final String xsdVersion) {
        if (!RP_BR_0110_IGNORE_VERSIONS.contains(xsdVersion) && "Other".equals(type)) {
            final Set<String> wrongValues = new HashSet<>(Arrays.asList("BIC", "IBAN", "OBAN"));
            if (other != null && wrongValues.contains(other.trim().toUpperCase())) {
                validationErrors.add(createValidationError(RP_BR_0110, messageRefId));
            }
        }
    }

    /**
     * Checks if the current reported payee is not a duplicate of a previously reported payee.
     * The check is done by comparing the set of non-empty names and account identifiers of the current reported payee
     * with the previously seen reported payees.
     *
     * @param validationErrors the list of validation errors
     * @param messageRefId     the message ref id
     * @param reportedPayees   the map of seen reported payees, maps the set of names and account identifiers to the doc ref id
     * @param names            the names of the payee
     * @param accounts         the account identifiers of the payee
     * @param docRefId         the doc ref id
     */
    public static void checkCmBr0150(
            final List<ValidationError> validationErrors,
            final String messageRefId,
            final HashMap<Set<XmlTypeAndValue>, String> reportedPayees,
            final List<XmlTypeAndValue> names,
            final List<XmlCountryTypeAndValue> accounts,
            final String docRefId
    ) {
        // Filter out empty names and account identifiers
        List<XmlTypeAndValue> filteredNames = names.stream()
                .filter(name -> isNotBlank(name.getValue()))
                .collect(Collectors.toList());
        List<XmlCountryTypeAndValue> filteredAccounts = accounts.stream()
                .filter(account -> isNotBlank(account.getValue()))
                .collect(Collectors.toList());

        // Set containing the filtered names and account identifiers of the reported payee,
        // which defines the uniqueness of a reported payee
        Set<XmlTypeAndValue> reportedPayee = Stream.concat(
                filteredNames.stream(),
                filteredAccounts.stream()
        ).collect(Collectors.toSet());

        // If at least one non-empty name and account identifier are present and the reported payee is already seen then CM_BR_0150
        if (!filteredNames.isEmpty() && !filteredAccounts.isEmpty() && reportedPayees.containsKey(reportedPayee)) {
            ValidationError error = createValidationError(CM_BR_0150, messageRefId, docRefId, null);
            String errorLongDesc = error.getErrorLongDesc();
            errorLongDesc = errorLongDesc.replace("<DocRefId1>", docRefId);
            errorLongDesc = errorLongDesc.replace("<DocRefId2>", reportedPayees.get(reportedPayee));
            error.setErrorLongDesc(errorLongDesc);
            validationErrors.add(error);
        } else {
            reportedPayees.put(reportedPayee, docRefId);
        }
    }

    /**
     * Custom error (99999).
     *
     * @param messageRefId the reference of the message
     * @param description  the description of the error
     * @return the validation error
     */
    public static ValidationError createCustomError(final String messageRefId, final String description) {
        final ValidationError validationError = createValidationError(CM_TR_9999, messageRefId);
        final ValidationErrorType validationErrorType = ValidationErrorTypeHolder.INSTANCE.findByCode(CM_TR_9999.getCode());
        final String longDescription = validationErrorType.getLongDescription() + "\n" + description;
        validationError.setErrorLongDesc(longDescription);
        return validationError;
    }

    /**
     * Create a {@link ValidationError} from an {@link ValidationErrorTypeEnum}.
     *
     * @param errorType    the validation error type
     * @param messageRefId the reference of the message
     * @return the validation error
     */
    public static ValidationError createValidationError(final ValidationErrorTypeEnum errorType, final String messageRefId) {
        return createValidationError(errorType, messageRefId, null, null);
    }

    /**
     * Create a {@link ValidationError} from an {@link ValidationErrorTypeEnum}.
     *
     * @param errorType             the validation error type
     * @param messageRefId          the reference of the message
     * @param docRefId              the doc ref id
     * @param transactionIdentifier the transaction identifier
     * @return the validation error
     */
    public static ValidationError createValidationError(final ValidationErrorTypeEnum errorType, final String messageRefId,
            final String docRefId, final String transactionIdentifier) {
        final ValidationErrorType validationErrorType = ValidationErrorTypeHolder.INSTANCE.findByCode(errorType.getCode());
        final ValidationError validationError = new ValidationError(errorType);
        validationError.setMessageRefId(messageRefId);
        validationError.setErrorCounter(1);
        validationError.setErrorShortDesc(validationErrorType.getDescription());
        validationError.setErrorLongDesc(validationErrorType.getLongDescription());
        validationError.setDocRefId(docRefId);
        validationError.setTransactionIdentifier(transactionIdentifier);
        return validationError;
    }

    /**
     * Convert greece string.
     *
     * @param code the code
     * @return the string
     */
    public static String convertGreece(final String code) {
        if (GREECE_GR.equalsIgnoreCase(code)) {
            return GREECE_EL;
        }
        return code;
    }
}
